<?php
/**
 * Elementor Widget
 * @package panthar
 * @since 1.0.0
 */

namespace Elementor;

class Panthar_Banner_Slider_Two_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'panthar-banner-slider-two-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Banner Slider Two', 'panthar-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['panthar_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        //banner address
        $this->add_control('address', [
            'label' => esc_html__('Address', 'panthar-core'),
            'type' => Controls_Manager::TEXT,
            'default' => 'Dewi Sri Street. 2891, Denpasar, Bali',
        ]);
        $this->add_control('mail', [
            'label' => esc_html__('Mail', 'panthar-core'),
            'type' => Controls_Manager::TEXT,
            'default' => 'info@example.com',
        ]);
        $this->add_control('phone', [
            'label' => esc_html__('Phone', 'panthar-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '+0 123 456 789',
        ]);

        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'panthar-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many course you want in masonry , enter -1 for unlimited course.')
        ]);
        $this->add_control('category', [
            'label' => esc_html__('Category', 'panthar-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => true,
            'options' => Panthar()->get_terms_names('case-study-cat', 'id'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'panthar-core'),
        ]);
        $this->add_control('order', [
            'label' => esc_html__('Order', 'panthar-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'panthar-core'),
                'DESC' => esc_html__('Descending', 'panthar-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'panthar-core')
        ]);
        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'panthar-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'panthar-core'),
                'title' => esc_html__('Title', 'panthar-core'),
                'date' => esc_html__('Date', 'panthar-core'),
                'rand' => esc_html__('Random', 'panthar-core'),
                'comment_count' => esc_html__('Most Comments', 'panthar-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'panthar-core')
        ]);
        $this->add_control(
            'items',
            [
                'label' => esc_html__('slidesToShow', 'panthar-core'),
                'type' => Controls_Manager::NUMBER,
                'description' => esc_html__('you can set how many item show in slider', 'panthar-core'),
                'default' => '3',
            ]
        );
        $this->add_control(
            'margin',
            [
                'label' => esc_html__('Margin', 'panthar-core'),
                'description' => esc_html__('you can set margin for slider', 'panthar-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 5,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'size_units' => ['px']
            ]
        );
        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
            ]
        );
        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
            ]
        );
        $this->add_control(
            'nav',
            [
                'label' => esc_html__('Nav', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'nav_left_arrow',
            [
                'label' => esc_html__('Nav Left Icon', 'panthar-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => [
                    'value' => 'fas fa-arrow-left',
                    'library' => 'solid',
                ],
                'condition' => ['nav' => 'yes']
            ]
        );
        $this->add_control(
            'nav_right_arrow',
            [
                'label' => esc_html__('Nav Right Icon', 'panthar-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => [
                    'value' => 'fas fa-arrow-right',
                    'library' => 'solid',
                ],
                'condition' => ['nav' => 'yes']
            ]
        );
        $this->add_control(
            'center',
            [
                'label' => esc_html__('Center', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),

            ]
        );
        $this->add_control(
            'hero_image_bg', [
                'label' => esc_html__('Background Image', 'panthar-core'),
                'type' => Controls_Manager::MEDIA,
                'show_label' => false,
                'description' => esc_html__('upload background image', 'panthar-core'),
            ]
        );
        $this->add_control(
            'autoplaytimeout',
            [
                'label' => esc_html__('Autoplay Timeout', 'panthar-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10000,
                        'step' => 2,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5000,
                ],
                'size_units' => ['px'],
                'condition' => array(
                    'autoplay' => 'yes'
                )
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'case_study_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'normal_background',
                'label' => esc_html__('Background', 'panthar-core'),
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .case-study-single-item-02 .content',
            ]
        );
        $this->add_control('name_color', [
            'label' => esc_html__('Name Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .case-study-single-item-02 .content .title" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('designation_color', [
            'label' => esc_html__('Designation Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .case-study-single-item-02 .content span" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('case_study_social_icon_styling_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);

        $this->start_controls_tabs(
            'case_study_social_icon_style_tabs'
        );

        $this->start_controls_tab(
            'case_study_social_icon_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'panthar-core'),
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'social_bg_icon_normal_background',
                'label' => esc_html__('Background', 'panthar-core'),
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .case-study-single-item-02 .social-icon li',
            ]
        );
        $this->add_control('social_icon_color', [
            'label' => esc_html__('Icon Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .case-study-single-item-02 .social-icon li" => "color: {{VALUE}}"
            ]
        ]);
        $this->end_controls_tab();

        $this->start_controls_tab(
            'case_study_social_icon_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'panthar-core'),
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'social_icon_hover_background',
                'label' => esc_html__('Background', 'panthar-core'),
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .case-study-single-item-02 .social-icon li:hover',
            ]
        );
        $this->add_control('social_hover_icon_color', [
            'label' => esc_html__('Icon Color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .case-study-single-item-02 .social-icon li:hover" => "color: {{VALUE}}"
            ]
        ]);

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control('case_study_typography_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'name_typography',
            'label' => esc_html__('Name Typography', 'panthar-core'),
            'selector' => "{{WRAPPER}} .case-study-single-item-02 .title"
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'designation_typography',
            'label' => esc_html__('Designation Typography', 'panthar-core'),
            'selector' => "{{WRAPPER}} .case-study-single-item-02 span"
        ]);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $navbar_top = '';
        $rand_numb = rand(333, 999999999);
        //slider settings
        $slider_settings = [
            "loop" => esc_attr($settings['loop']),
            "margin" => esc_attr($settings['margin']['size'] ?? 0),
            "items" => esc_attr($settings['items'] ?? 1),
            "center" => esc_attr($settings['center']),
            "autoplay" => esc_attr($settings['autoplay']),
            "autoplaytimeout" => esc_attr($settings['autoplaytimeout']['size'] ?? 0),
            "nav" => esc_attr($settings['nav']),
            "navleft" => panthar_core()->render_elementor_icons($settings['nav_left_arrow']),
            "navright" => panthar_core()->render_elementor_icons($settings['nav_right_arrow'])
        ];
        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        //setup query
        $args = array(
            'post_type' => 'service',
            'posts_per_page' => $total_posts,
            'order' => $order,
            'orderby' => $orderby,
            'post_status' => 'publish'
        );

        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => 'service-cat',
                    'field' => 'term_id',
                    'terms' => $category
                )
            );
        }
        $post_data = new \WP_Query($args);
        ?>
        <div class="banner-two-wrap">
            <?php
                $hero_image_bg = $this->get_settings('hero_image_bg');
            ?>
            <div class="banner-2-bg" style="background-image: url(<?php echo esc_url($hero_image_bg['url']); ?>);"></div>
            <div class="container custom-container">
                <div class="info-bar-address">
                    <ul>
                        <?php if(!empty($settings['address'])) : ?>
                            <li><i class="icomoon-map-marker"></i><?php echo esc_html($settings['address']); ?></li>
                        <?php endif; ?>
                        <?php if(!empty($settings['mail'])) : ?>
                            <li><a href="mailto:<?php echo esc_url($settings['mail']); ?>"><i class="icomoon-mail"></i><?php echo esc_html($settings['mail']); ?></a></li>
                        <?php endif; ?>
                    </ul>
                </div>
                <?php if(!empty($settings['phone'])) : ?>
                    <div class="info-bar-phone">
                        <ul>
                            <li>
                                <a href="tel:<?php echo esc_html($settings['phone']); ?>">
                                    <?php echo esc_html('For', 'panthar') ?> <span><?php echo esc_html('Emergency!', 'panthar') ?></span> <?php echo esc_html($settings['phone']); ?>
                                </a>
                            </li>
                        </ul>
                    </div>
                <?php endif; ?>
                <div class="case-study-carousel-wrap panthar-rtl-slider">
                    <div class="case-study-carousel"
                         id="case-study-one-carousel-<?php echo esc_attr($rand_numb); ?>"
                         data-settings='<?php echo json_encode($slider_settings) ?>'
                    >
                        <?php
                        while ($post_data->have_posts()) : $post_data->the_post();
                            $post_id = get_the_ID();
                            $img_id = get_post_thumbnail_id($post_id) ? get_post_thumbnail_id($post_id) : false;
                            $img_url_val = $img_id ? wp_get_attachment_image_src($img_id, 'panthar_service_banner', false) : '';
                            $img_url = is_array($img_url_val) && !empty($img_url_val) ? $img_url_val[0] : '';
                            $img_alt = get_post_meta($img_id, '_wp_attachment_image_alt', true);
                            ?>
                            <div class="banner-service-inner" style="background-image: url(<?php echo esc_url($img_url)?>)">
                                <a class="read-more" href="<?php the_permalink(); ?>">
                                    <i class="fas fa-arrow-right"></i>
                                </a>
                                <div class="details">
                                    <span class="animate-img-2"></span>
                                    <a href="<?php the_permalink(); ?>">
                                        <h4 class="title"><?php echo esc_html(get_the_title($post_id)) ?></h4>
                                    </a>
                                </div>
                            </div>
                        <?php endwhile; ?>
                    </div>

                    <?php if (!empty($settings['nav'])) : ?>
                        <div class="slick-carousel-controls banner-service-control <?php echo $navbar_top; ?>">
                            <div class="slider-nav"></div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

    <?php }
}

Plugin::instance()->widgets_manager->register(new Panthar_Banner_Slider_Two_Widget());