<?php
/**
 * Elementor Widget
 * @package panthar
 * @since 1.0.0
 */

namespace Elementor;

class Panthar_gallery_Slider_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'panthar-gallery-slider-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Gallery Slider', 'panthar-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'eicon-person';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['panthar_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {
        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('General Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();
        $repeater->add_control(
            'gallery_image', [
                'label' => esc_html__('Background Image', 'panthar-core'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'show_label' => false,
                'description' => esc_html__('upload background image', 'panthar-core'),
            ]
        );
        $this->add_control('gallery_items', [
            'label' => esc_html__('Gallery Slider Item', 'panthar-core'),
            'type' => Controls_Manager::REPEATER,
            'fields' => $repeater->get_controls(),
            'default' => [
                [
                    'image' => array(
                        'url' => Utils::get_placeholder_image_src()
                    )
                ]
            ],

        ]);
        $this->end_controls_section();

        $this->start_controls_section(
            'slider_settings_section',
            [
                'label' => esc_html__('Slider Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'items',
            [
                'label' => esc_html__('slidesToShow', 'panthar-core'),
                'type' => Controls_Manager::NUMBER,
                'description' => esc_html__('you can set how many item show in slider', 'panthar-core'),
                'default' => '4',
            ]
        );
        $this->add_control(
            'margin',
            [
                'label' => esc_html__('Margin', 'panthar-core'),
                'description' => esc_html__('you can set margin for slider', 'panthar-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 5,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 0,
                ],
                'size_units' => ['px']
            ]
        );
        $this->add_control(
            'loop',
            [
                'label' => esc_html__('Loop', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
            ]
        );
        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
            ]
        );
        $this->add_control(
            'progress_bar',
            [
                'label' => esc_html__('Slider Progress', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'nav',
            [
                'label' => esc_html__('Nav', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'dots',
            [
                'label' => esc_html__('Dots', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'nav_left_arrow',
            [
                'label' => esc_html__('Nav Left Icon', 'panthar-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => [
                    'value' => 'fas fa-arrow-left',
                    'library' => 'solid',
                ],
                'condition' => ['nav' => 'yes']
            ]
        );
        $this->add_control(
            'nav_right_arrow',
            [
                'label' => esc_html__('Nav Right Icon', 'panthar-core'),
                'type' => Controls_Manager::ICONS,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),
                'default' => [
                    'value' => 'fas fa-arrow-right',
                    'library' => 'solid',
                ],
                'condition' => ['nav' => 'yes']
            ]
        );
        $this->add_control(
            'center',
            [
                'label' => esc_html__('Center', 'panthar-core'),
                'type' => Controls_Manager::SWITCHER,
                'description' => esc_html__('you can set yes/no to enable/disable', 'panthar-core'),

            ]
        );
        $this->add_control(
            'autoplaytimeout',
            [
                'label' => esc_html__('Autoplay Timeout', 'panthar-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10000,
                        'step' => 2,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5000,
                ],
                'size_units' => ['px'],
                'condition' => array(
                    'autoplay' => 'yes'
                )
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'gallery_member_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'panthar-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control('name_color', [
            'label' => esc_html__('popup_btn_color', 'panthar-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .gallery-popup-btn-relative .gallery-popup-btn" => "background: {{VALUE}}"
            ]
        ]);

        $this->end_controls_tab();

        $this->add_control('gallery_typography_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'name_typography',
            'label' => esc_html__('Name Typography', 'panthar-core'),
            'selector' => "{{WRAPPER}} .gallery-single-item-02 .title"
        ]);
        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'designation_typography',
            'label' => esc_html__('Designation Typography', 'panthar-core'),
            'selector' => "{{WRAPPER}} .gallery-single-item-02 span"
        ]);
        $this->end_controls_section();

    }


    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        $navbar_top = '';
        if($settings['dots'] == ''){
            $navbar_top = 'nav-style-top';
        }
        $rand_numb = rand(333, 999999999);
        
        //slider settings
        $slider_settings = [
            "loop" => esc_attr($settings['loop']),
            "margin" => esc_attr($settings['margin']['size'] ?? 0),
            "items" => esc_attr($settings['items'] ?? 1),
            "center" => esc_attr($settings['center']),
            "autoplay" => esc_attr($settings['autoplay']),
            "autoplaytimeout" => esc_attr($settings['autoplaytimeout']['size'] ?? 0),
            "nav" => esc_attr($settings['nav']),
            "dot" => esc_attr($settings['dots']),
            "navleft" => panthar_core()->render_elementor_icons($settings['nav_left_arrow']),
            "navright" => panthar_core()->render_elementor_icons($settings['nav_right_arrow'])
        ];

        ?>
        <div class="gallery-member-carousel-wrap panthar-rtl-slider">
            <div class="gallery-member-carousel"
                 id="gallery-member-one-carousel-<?php echo esc_attr($rand_numb); ?>"
                 data-settings='<?php echo json_encode($slider_settings) ?>'
            >   
                <?php foreach ($settings['gallery_items'] as $item): ?>
                    <div class="thumb-wrap">
                        <div class="thumb gallery-popup-btn-relative">
                            <a class="gallery-popup-btn" href="<?php echo esc_url($item['gallery_image']['url']); ?>"><i class="fa fa-plus"></i></a>
                            <img src="<?php echo esc_url($item['gallery_image']['url']); ?>" alt="<?php echo esc_attr($item['gallery_image']['url']); ?>">
                            
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php if (!empty($settings['nav'] || $settings['dots'])) : ?>
                <div class="slick-carousel-controls <?php echo $navbar_top; ?>">
                    <div class="container"> 
                        <?php if (!empty($settings['dots'])) : ?>
                            <div class="slide-dot">
                                <div class="slider-dots"></div>
                            </div>
                        <?php endif; ?>
                        <?php if (!empty($settings['nav'])) : ?>
                            <div class="slide-nav">
                                <div class="slider-nav"></div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register(new Panthar_gallery_Slider_Widget());